function [adj_matrix, node_features, edge_features_norm] = generate_graph(mpc_fault, p_sampled, q_sampled)
    % generate_graph - 生成图数据，包括邻接矩阵、节点特征和边特征
    %
    % 输入参数:
    %   mpc_fault   - MATPOWER的故障后案例结构体
    %   p_sampled   - 采样后的有功功率向量（长度为母线数量）
    %   q_sampled   - 采样后的无功功率向量（长度为母线数量）
    %
    % 输出参数:
    %   adj_matrix         - 邻接矩阵（稀疏矩阵）
    %   node_features      - 节点特征矩阵 [PD_norm, QD_norm, BUS_TYPE_norm, Pg_max_norm, Qg_max_norm, Pg_min_norm, Qg_min_norm, Pg_ratio_norm, Qg_ratio_norm]
    %   edge_features_norm - 边特征矩阵 [R_norm, X_norm]
    
    % 定义常量
    define_constants;
    
    % 提取母线、支路和发电机数据
    bus = mpc_fault.bus;
    branches = mpc_fault.branch;
    gens = mpc_fault.gen;  % 从 mpc_fault 中提取发电机数据
    
    % 获取母线数量
    num_buses = size(bus, 1);
    
    % 构建邻接矩阵，使用稀疏矩阵提高效率
    adj_matrix = sparse(num_buses, num_buses);
    
    % 仅考虑状态为1的支路
    active_branches = branches(branches(:, BR_STATUS) == 1, :);
    
    % 构建邻接矩阵
    for i = 1:size(active_branches, 1)
        from = active_branches(i, F_BUS);
        to = active_branches(i, T_BUS);
        adj_matrix(from, to) = 1;
        adj_matrix(to, from) = 1;  % 无向图
    end
    
    % 初始化节点特征
    % 基础特征：有功需求 PD， 无功需求 QD， 母线类型 BUS_TYPE
    feature_PD = p_sampled;
    feature_QD = q_sampled;
    feature_TYPE = bus(:, BUS_TYPE);
    
    % 使用单位向量归一化 BUS_TYPE
    feature_TYPE_norm = normalize_feature_unit_vector(feature_TYPE);
    
    % 初始化发电机特征
    Pg_max = zeros(num_buses, 1);
    Qg_max = zeros(num_buses, 1);
    Pg_min = zeros(num_buses, 1);
    Qg_min = zeros(num_buses, 1);
    
    % 计算每个母线的发电机特征
    for bus_num = 1:num_buses
        % 找出连接到当前母线的所有在线发电机
        connected_gens = gens(gens(:, GEN_BUS) == bus_num & gens(:, GEN_STATUS) == 1, :);
        
        if ~isempty(connected_gens)
            Pg_max(bus_num) = max(connected_gens(:, PMAX));
            Qg_max(bus_num) = max(connected_gens(:, QMAX));
            Pg_min(bus_num) = min(connected_gens(:, PMIN));
            Qg_min(bus_num) = min(connected_gens(:, QMIN));
        else
            % 没有在线发电机，设为0
            Pg_max(bus_num) = 0;
            Qg_max(bus_num) = 0;
            Pg_min(bus_num) = 0;
            Qg_min(bus_num) = 0;
        end
    end
    
    % 计算 Pg_min / Pg_max 和 Qg_min / Qg_max
    Pg_ratio = zeros(num_buses, 1);
    Qg_ratio = zeros(num_buses, 1);
    
    % 处理除以零的情况
    ratio_Pg = Pg_max ~= 0;
    ratio_Qg = Qg_max ~= 0;
    
    Pg_ratio(ratio_Pg) = Pg_min(ratio_Pg) ./ Pg_max(ratio_Pg);
    Qg_ratio(ratio_Qg) = Qg_min(ratio_Qg) ./ Qg_max(ratio_Qg);
    
    % 使用 Min-Max 归一化
    % feature_PD_norm = normalize_feature_min_max(feature_PD);
    % feature_QD_norm = normalize_feature_min_max(feature_QD);
    % Pg_max_norm = normalize_feature_min_max(Pg_max);
    % Qg_max_norm = normalize_feature_min_max(Qg_max);
    % Pg_min_norm = normalize_feature_min_max(Pg_min);
    % Qg_min_norm = normalize_feature_min_max(Qg_min);
    % Pg_ratio_norm = normalize_feature_min_max(Pg_ratio);
    % Qg_ratio_norm = normalize_feature_min_max(Qg_ratio);
    
    % 组合节点特征
    % node_features = [feature_PD_norm, feature_QD_norm, feature_TYPE_norm, ...
    %                  Pg_max_norm, Qg_max_norm, Pg_min_norm, Qg_min_norm, ...
    %                  Pg_ratio_norm, Qg_ratio_norm];
    node_features = [feature_PD, feature_QD, feature_TYPE_norm, ...
                     Pg_max, Qg_max, Pg_min, Qg_min, ...
                     Pg_ratio, Qg_ratio];
    
    % 定义边特征
    if ~isempty(active_branches)
        R = active_branches(:, BR_R);
        X = active_branches(:, BR_X);
        edge_features = [R, X];
        % 归一化边特征为单位向量
        edge_features_norm = [normalize_feature_unit_vector(R), normalize_feature_unit_vector(X)];
    else
        edge_features_norm = [];
    end
end

function norm_feat = normalize_feature_min_max(feat)
    % normalize_feature_min_max - 将特征向量归一化到 [0, 1] 使用最小-最大归一化
    %
    % 输入:
    %   feat - 特征向量
    %
    % 输出:
    %   norm_feat - 归一化后的特征向量
    
    min_val = min(feat);
    max_val = max(feat);
    if max_val ~= min_val
        norm_feat = (feat - min_val) / (max_val - min_val);
    else
        norm_feat = zeros(size(feat));
    end
end

function norm_feat = normalize_feature_unit_vector(feat)
    % normalize_feature_unit_vector - 将特征向量归一化为单位向量（L2 归一化）
    %
    % 输入:
    %   feat - 特征向量
    %
    % 输出:
    %   norm_feat - 归一化后的单位向量
    
    vec_norm = norm(feat, 2);  % 计算 L2 范数
    if vec_norm ~= 0
        norm_feat = feat / vec_norm;
    else
        norm_feat = zeros(size(feat));  % 如果向量的 L2 范数为 0，返回零向量
    end
end
